#include <set>
#include <map>

#include <blitz/array.h>
#include <gnuplot-iostream.h>

#include "../common/hdf5.hpp"

int main()
{
  // simulations                                                                
  std::map<double, std::map<bool, H5::H5File>> cases;     
  for (auto &mlt_t : std::set<double>{1.,200})
  {
    for (auto &khl : std::set<bool>{1,0}) 
    { 
      std::ostringstream fname;                                                 
      fname << "refcase_khl=" << khl << "_t_x_" << mlt_t << ".nc";         
      {                                                                         
	std::string cmd = "cp ../common/refcase.nc " + fname.str();             
	system(cmd.c_str());                                                    
      }                                                                         
      {                                                                         
	H5::H5File file(fname.str().c_str(), H5F_ACC_RDWR);                     
	h5_setpar(file, "kelvin", double(khl));
	h5_setpar(file, "raoult", double(khl));
	h5_setpar(file, "RH0", 1.-1e-6);
	h5_setpar(file, "reltol", 1e-8);
	h5_setpar(file, "n_cycl", 1.);
	h5_setpar(file, "t_hlf", mlt_t * h5_getpar(file, "t_hlf"));
      }                                                                         
      {                                                                         
	std::string cmd = "../../main " + fname.str();
	system(cmd.c_str());                                                    
      }                                                                         
      cases[mlt_t][khl] = H5::H5File(fname.str().c_str(), H5F_ACC_RDONLY);     
    }
  }

  // plotting
  Gnuplot gp;
  gp << "set term svg dynamic\n";
  gp << "set output 'nokhlr.svg'\n";
  gp << "set grid\n";
  gp << "set linetype 2 dashtype (1.75,.75)\n";
  gp << "set key top samplen .5 invert\n";
  gp << "set ylabel 'displacement [z]'\n";
  gp << "set yrange [-10:160]\n";
  gp << "set xrange [-.125:.225]\n";
  gp << "set xlabel 'S=RH-1 [%]'\n";
  gp << "set multiplot layout 1,2\n";

  gp << "set label 1 'ascent          ' at graph .05,.04\n";                
  gp << "set label 2 '            / descent' at graph .05,.04 textcolor rgb 'orange'\n";

  double lw0=6, dlw=3;

  gp << "plot 1./0 not";
  double lw = lw0;
  for (auto &c_w : cases)
  {
    for (auto &cs : c_w.second)
    {
      auto &h5 = cs.second;
      bool khl = cs.first;

      double                                                                
	t_hlf = h5_getpar(h5, "t_hlf");
      gp << ", '-' using (100*($1-1)):($3<=" << 1.05*t_hlf << "?$2:1./0) with lines lt " << (khl?1:2) << " lw " << lw << " lc rgb 'black' notitle";
      gp << ", '-' using (100*($1-1)):($3>=" << 1.000*t_hlf << "?$2:1./0) with lines lt " << (khl?1:2) << " lw " << .5*lw << " lc rgb 'orange' notitle";
    }
    lw -= dlw;
  }
  gp << "\n";

  for (auto &c_w : cases)
  {
    for (auto &cs : c_w.second)
    {
      auto &h5 = cs.second;
      std::array<blitz::Array<float,1>, 3> data;
      data[0].reference(h5_getarr(h5, "RH"));
      data[1].reference(h5_getarr(h5, "z"));
      data[2].reference(h5_getarr(h5, "t"));
      for (int i=0; i < 2; ++i) gp.send(data);
    }
  }

  gp << "set xlabel 'wet radius'\n";
  gp << "set ylabel 'S=RH-1 [%]'\n";
  gp << "set yrange [-.125:.225]\n";
  gp << "set xrange [*:*]\n";
  gp << "plot 1./0 not";
  lw=lw0;
  for (auto &c_w : cases)
  {
    for (auto &cs : c_w.second)
    {
      bool khl = cs.first;
      auto &h5 = cs.second;

      double 
        t_hlf = h5_getpar(h5, "t_hlf"),
	z_hlf = h5_getpar(h5, "z_hlf");                                     
      std::ostringstream w_mean;                                            
      w_mean << "<w>=" << (100 * z_hlf / t_hlf) << " cm/s"; 

      gp << ", '-' using (1e6*$1):($3<=" << 1.05*t_hlf << "?100*($2-1):1./0) with lines lt " << (khl?1:2) << " lw " << lw << " lc rgb 'black' title '" << (khl?w_mean.str():"Köhler: off") << "'";
      gp << ", '-' using (1e6*$1):($3>=" << 1.000*t_hlf << "?100*($2-1):1./0) with lines lt " << (khl?1:2) << " lw " << .5 * lw << " lc rgb 'orange' notitle";
    }
    lw-=dlw;
  }
  gp << "\n";

  for (auto &c_w : cases)
  {
    for (auto &cs : c_w.second)
    {
      auto &h5 = cs.second;
      std::array<blitz::Array<float,1>, 3> data;
      data[0].reference(h5_getarr(h5, "rw1"));
      data[1].reference(h5_getarr(h5, "RH"));
      data[2].reference(h5_getarr(h5, "t"));
      for (int i=0; i<2; ++i) gp.send(data);
    }
  }
}
