#include <string>
#include <stdexcept>
#include <set>

#include <blitz/array.h>
#include <gnuplot-iostream.h>

#include "../common/hdf5.hpp"

int main()
{
  const double mlt_t = 500;

  Gnuplot gp;
  gp << "set encoding utf8\n";
  gp << "set term svg dynamic enhanced\n";
  gp << "set output 'cspcat.svg'\n";
  //gp << "set multiplot layout 1,2\n";

//  gp << "set grid xtics, ytics, mxtics, mytics\n";
  gp << "set xrange [.08:13.5]\n";
  gp << "set xtics 5\n";
  gp << "set ytics 25\n";
  gp << "set mztics 4\n";

  gp << "set nomxtics\n";
  //gp << "set mztics 25\n";
  //gp << "set mytics 1\n";

  gp << "set view 45,285\n";
  gp << "set border 63-16\n";
  gp << "set grid xtics ytics ztics mztics\n";
  gp << "set ticslevel 0\n";

  gp << "set label 3 'ascent          ' at screen .68,.88\n";
  gp << "set label 4 '             / descent' at screen .68,.88 textcolor rgb 'orange'\n";

  for (auto &kvec : std::list<std::pair<std::string,std::list<double>>>{
    {"rd1",    std::list<double>{pow(1.39, -3), pow(1.39,-2), pow(1.39,-1), 1., pow(1.39,1), pow(1.39,2), pow(1.39,3), pow(1.39,4)}}, 
    {"N1_stp", std::list<double>{.001, .01, .1, 1., 10.}},
  }) 
  {
    // simulations
    std::list<H5::H5File> cases;

    const auto &mlt_key = kvec.first;
    for (auto &mlt_val : kvec.second)
    {
      const auto 
        &mlt_r = (mlt_key == "rd1"    ? mlt_val : 1.),
        &mlt_n = (mlt_key == "N1_stp" ? mlt_val : 1.);

      std::ostringstream fname;
      fname << "refcase_rd_x_" << mlt_r << "_n_x_" << mlt_n << ".nc";
      {
	std::string cmd = "cp ../common/refcase.nc " + fname.str(); 
	system(cmd.c_str());
      }
      {
	H5::H5File file(fname.str().c_str(), H5F_ACC_RDWR);
	h5_setpar(file, "t_hlf",  mlt_t * h5_getpar(file, "t_hlf"));
	h5_setpar(file, "reltol", 1e0 * h5_getpar(file, "reltol"));

	h5_setpar(file, "rd1",    mlt_r * h5_getpar(file, "rd1"));
	h5_setpar(file, "N1_stp", mlt_n * h5_getpar(file, "N1_stp"));
      }
      {
	std::string cmd = "../../main " + fname.str(); 
	system(cmd.c_str());
      }
      cases.push_back(H5::H5File(fname.str().c_str(), H5F_ACC_RDONLY));
    }

    double lw0 = 5, dlw = .5;

    // plotting
    gp << "set logscale xz\n";
    gp << "set xlabel 'wet radius [µm]'  rotate parallel\n";
    gp << "set ylabel 'displacement [m]'\n";
    gp << "set zlabel 'dry radius [µm]'  rotate parallel\n";
    gp << "set yrange [25:75]\n";
    gp << "splot 1./0 notitle";
    double lw = lw0;
    for (auto &h5 : cases)
    { 
      double 
	mlt = (mlt_key == "rd1" ? 1e6 : 1e-6) * h5_getpar(h5, mlt_key),
	t_hlf = h5_getpar(h5, "t_hlf");

      gp
	<< ",'-' using ($1*1e6):($3<=" << 1.05*t_hlf << "?$2:1./0):(" << mlt << ") with lines lc rgb 'black' lw " << lw << " notitle"
	<< ",'-' using ($1*1e6):($3>=" <<      t_hlf << "?$2:1./0):(" << mlt << ") with lines lc rgb 'orange'  lw " << lw/3. << " notitle"
      ;
      lw-=dlw;
    }
    gp << "\n";

    for (auto &h5 : cases)
    {
      std::array<decltype(h5_getarr(h5, "t")), 3> data;
      data[0].reference(h5_getarr(h5, "rw1"));
      data[1].reference(h5_getarr(h5, "z"));
      data[2].reference(h5_getarr(h5, "t"));
      for (int i=0; i<2; ++i) gp.send(data);
    }
break;
  }

/*
    double lw0 = 7., dlw = 1.05;
    {
      gp << "set key bottom right samplen .5\n";

      gp << "set xlabel 'S=RH-1 [%]' offset 0,.5\n";
      gp << "set ylabel 'displacement [m]' offset 1.5,0\n";
      gp << "set xtics .5\n";
      gp << "set ytics 50\n";
      gp << "set xrange [-.92:.92]\n";
      gp << "set yrange [0:160]\n";

      {
	std::ostringstream tmp;
	tmp << 1e6*h5_getpar(cases.begin()->second, "rd1");
	gp << "set label 1 'r_d = " << tmp.str() << " µm' at graph .05,.85\n";
      }
      {
	std::ostringstream tmp;
	tmp << 1e-6*h5_getpar(cases.begin()->second, "N1_stp");
	gp << "set label 2 'N_{STP} = " << tmp.str() << " cm^{-3}' at graph .05,.90\n";
      }

      gp << "plot 1./0 notitle";
      {
	int lw=lw0;
	for (auto &cs : cases)
	{
	  auto &h5 = cs.second;
	  double
	    t_hlf = h5_getpar(h5, "t_hlf"), 
	    z_hlf = h5_getpar(h5, "z_hlf");
	  std::ostringstream w_mean;
	  w_mean << (100 * z_hlf / t_hlf);
	  gp
	    << ",'-' using (100*($1-1)):($3<=" << 1.05*t_hlf << "?$2:1./0) with lines lc rgb 'black' lw " << lw << " title '<w>=" << w_mean.str() << " cm/s'"
	    << ",'-' using (100*($1-1)):($3>=" << t_hlf << "?$2:1./0) with lines lc rgb 'orange' lw " << lw/3. << " notitle";
	  lw-=dlw;
	}
      }
      gp << "\n";
      for (auto &cs : cases)
      {
	auto &h5 = cs.second;
	std::array<decltype(h5_getarr(h5, "t")), 3> data;
	data[0].reference(h5_getarr(h5, "RH"));
	data[1].reference(h5_getarr(h5, "z"));
	data[2].reference(h5_getarr(h5, "t"));
	for (int i=0; i<2; ++i) gp.send(data);
      }

      gp << "set ylabel 'S=RH-1 [%]' offset 1.5,0\n";
      gp << "set xlabel 'wet radius [µm]' offset 0,.5\n";
      gp << "set ytics .5\n";
      gp << "set xrange [.08:13.5]\n";
      gp << "set logscale x\n";
      gp << "set yrange [-.92:.92]\n";
      gp << "plot 1./0 notitle lc rgb 'black'";

      {
	int lw=lw0;
	for (auto &cs : cases)
	{ 
	  auto &h5 = cs.second;
	  double t_hlf = h5_getpar(h5, "t_hlf");
	  if (lw == lw0) 
	    gp
	      << ",'-' using ($1*1e6):(100*($4-1)) with lines lc rgb 'grey' lw " << 1.75*lw0 << " title 'Köhler curve'"
	    ;
	  gp
	    << ",'-' using ($1*1e6):($3<=" << 1.05*t_hlf << "?100*($2-1):1./0) with lines lc rgb 'black' lw " << lw << " notitle"
	    << ",'-' using ($1*1e6):($3>=" << t_hlf << "?100*($2-1):1./0) with lines lc rgb 'orange'  lw " << lw/3. << " notitle"
	  ;
	  lw-=dlw;
	}
      }
      gp << "\n";

      bool wbg=true;
      for (auto &cs : cases)
      {
	auto &h5 = cs.second;
	std::array<decltype(h5_getarr(h5, "t")), 4> data;
	data[0].reference(h5_getarr(h5, "rw1"));
	data[1].reference(h5_getarr(h5, "RH"));
	data[2].reference(h5_getarr(h5, "t"));
	data[3].reference(h5_getarr(h5, "RHeq1"));
	for (int i=0; i<(wbg?3:2); ++i) gp.send(data);
	wbg=false;
      }
    }

*/
}
