#include <algorithm>
#include <limits>

#include "src/solver.hpp"
#include "src/odeset.hpp"
#include "src/invfft.hpp"                                            
#include "src/zintrp.hpp"
#include "src/hdf5io.hpp"

template <typename xi_t>
void run(hdf5io_t &hdf5io)
{
  using odeset_t = odeset_t<xi_t>;
  struct params_t : 
    zintrp_t::params_t,
    solver_t<odeset_t>::params_t,
    invfft_t::params_t,
    odeset_t::params_t
  {
    params_t(const hdf5io_t &io) :
      zintrp_t::params_t({
	.n_cycl = (int)io.getpar("n_cycl"), 
      }),
      solver_t<odeset_t>::params_t({
	.reltol = io.getpar("reltol"), 
	.abstol = io.getpar("abstol")
      }),
      invfft_t::params_t({
	.z_hlf  = io.getpar("z_hlf") * si::metres,
	.t_hlf  = io.getpar("t_hlf") * si::seconds,
	.freq   = io.getpar("freq")  * si::hertz,
	.ampl   = io.getpar("ampl")  * si::metres
      }),
      odeset_t::params_t({
	.p0     = io.getpar("p0") * si::pascals,
	.T0     = io.getpar("T0") * si::kelvins,
	.RH0    = io.getpar("RH0") * si::dimensionless(),
	.N1_stp = io.getpar("N1_stp") / si::cubic_metres,
	.N2_stp = io.getpar("N2_stp") / si::cubic_metres,
	.kpa1   = io.getpar("kpa1") * si::dimensionless(),
	.kpa2   = io.getpar("kpa2") * si::dimensionless(),
	.rd3_1  = pow(io.getpar("rd1"), 3) * si::cubic_metres,
	.rd3_2  = pow(io.getpar("rd2"), 3) * si::cubic_metres,
        .kelvin = (bool)io.getpar("kelvin"),
        .raoult = (bool)io.getpar("raoult")
      })
    {}
  } params(hdf5io);

  auto zintrp = zintrp_t(invfft_t()(params), params);
  auto odeset = odeset_t(zintrp, params);
  auto solver = solver_t<odeset_t>(odeset, params);

  while (solver.t < params.n_cycl * 2 * (params.t_hlf / si::seconds))
  {
    std::array<double, hdf5io_t::n_vars> rec;
    rec[hdf5io_t::ix_t ] = solver.t;
    rec[hdf5io_t::ix_z ] = zintrp.z(solver.t);
    auto diag = odeset.diag(solver.state, solver.deriv, solver.t);
    rec[hdf5io_t::ix_RH] = diag.RH;
    rec[hdf5io_t::ix_T] = diag.T;
    rec[hdf5io_t::ix_RHeq1] = diag.RHeq1;
    rec[hdf5io_t::ix_RHeq2] = diag.RHeq2;
    rec[hdf5io_t::ix_rw1] = diag.rw1;
    rec[hdf5io_t::ix_rw2] = diag.rw2;
    rec[hdf5io_t::ix_rc1] = diag.rc1;
    rec[hdf5io_t::ix_rc2] = diag.rc2;
    rec[hdf5io_t::ix_rt] = diag.rt;
    rec[hdf5io_t::ix_rv] = diag.rv;
    rec[hdf5io_t::ix_th] = diag.th;
    rec[hdf5io_t::ix_dot_th] = diag.dot_th;
    rec[hdf5io_t::ix_dot_rv] = diag.dot_rv;
    rec[hdf5io_t::ix_dot_rw1] = diag.dot_rw1;
    rec[hdf5io_t::ix_dot_rw2] = diag.dot_rw2;
    hdf5io.putrec(rec);

    solver.step();
  }
}

int main(int argc, char **argv)
{
  if (!(argc == 2 || argc == 3))
    throw std::runtime_error("expecting one or two arguments - HDF5 filename & xi variable choice (id,p2,p3,ln)");

  hdf5io_t hdf5io(argv[1]);
  
  if (argc == 2 || std::string(argv[2]) == "id")
    run<xi_id<>>(hdf5io);
  else if (std::string(argv[2]) == "p2")
    run<xi_p2<>>(hdf5io);
  else if (std::string(argv[2]) == "p3")
    run<xi_p3<>>(hdf5io);
  else if (std::string(argv[2]) == "ln")
    run<xi_ln<>>(hdf5io);
  else
    throw std::runtime_error("unknown xi choice (second argument)");
}
